function y = advection_ell2(N,n,tmax,ex,plt)
% Verified computations for 1d advection equation with variable
% coefficients using C_0-semigroup on the ell^2 sequence space.
% 
% Inputs
% N: # of Fourier basses
% n: # of Chebyshev basses
% tmax: end time, i.e., t in [0,tmax]
% ex = 1: Example 1
%    = 2: Example 2
%    = 3: Example 3
% plt = 1: plot approximate solution (without verification)
% 
% Outputs
%    y(1)=N;
%    y(2)=n;
%    y(3)=tmax;
%    y(4)=initerr;
%    y(5)=residual;
%    y(6)=error;
%    y(7)=app_time;
%    y(8)=rig_time;

% 2018.7.15 written by A. Takayasu

tic;
%Property of Chebfun
chebfunpref.setDefaults('factory');
chebfunpref.setDefaults('fixedLength',n);
%Property of ODE45
opts = odeset('abstol',10^(-17),'reltol',10^(-17));

%Default Variable
h = 2*pi/(2*N-1);
x = h*(0:2*N);
k = (-N:N)';

if ex==1
    %Example 1
    % Calculate Init error
    a_k=exp(-k.^2/intval(400)-(1i*k))/(20*sqrt(intval('pi')));
%     initerr = norm(rad(a_k),2)+0.5*sqrt(1-erf(intval(N)/10/intval('sqrt2')));
    initerr = norm(rad(a_k),2)+0.5*sqrt(erfc(intval(N)/10/intval('sqrt2')));
    % Set omega, Core part of convolution matrix, Tail of sequence c
    omega=intval(0.5);
    real_c_hat=[zeros(1,N-2) ...
        (-1i*sin(intval(2))-cos(intval(2)))/4 ...k=-2
        0 intval('1.01') 0 ...k=0
        (1i*sin(intval(2))-cos(intval(2)))/4 ...k=2
        zeros(1,N-2)]; %c_hat length = 2N+1
    c_inf=0;
elseif ex==2
    %Example 2
    % Calculate Init error
    a_k=intval((-2).^-abs(k));
    initerr = intval('sqrt2')/intval('sqrt3')*(sqrt(0.5)^N);
    % Set omega, Core part of convolution matrix, Tail of sequence c
    omega=intval('0.49');
    real_c_hat = [zeros(1,N-2)...
        intval('0.49')/2 ...k=-2
        0 intval(1) 0 ...k=0
        intval('0.49')/2 zeros(1,N-2)];%k=2
    c_inf=0;
elseif ex==3
    %Example 3
    % Calculate Init error
    a_k=intval((-2).^-abs(k));
    initerr = intval('sqrt2')/intval('sqrt3')*(sqrt(0.5)^N);
    % Set omega, Core part of convolution matrix, Tail of sequence c
    omega=intval('0.64');
    real_c_hat = [zeros(1,N-3)...
        intval('0.3')/(2*1i) ...k=-3
        -intval('0.19')/2 ...k=-2
        0 -intval(1) 0 ...k=0
        -intval('0.19')/2 ...k=2
        -intval('0.3')/(2*1i) zeros(1,N-3)];%k=3
    c_inf=0;
else
    disp('4th argument should be 1, 2, or 3.')
    chebfunpref.setDefaults('factory');
    return
end

% Build N-length Convolution Matrix
zero_to_k = ([real_c_hat(N+1:end) zeros(1,3*N)]);
zero_to_mk = ([fliplr(real_c_hat(1:N+1)) zeros(1,3*N)]);

T = toeplitz(zero_to_k,zero_to_mk);    %toeplitz(first_col, first_row)
clip_T = T(:,N+1:3*N+1);

% Core for -N:N, beside for -2N:N-1, N+1:2N
convmat_core = clip_T(N+1:3*N+1,:);
convmat_beside = [clip_T(1:N,:); zeros(1,2*N+1) ; clip_T(3*N+2:end,:)];

% Compute approximate function by Chebfun ode45
app_time = toc;
nofft=chebfun.ode45(@(t,y) -mid(convmat_core)*(1i*(k.*y)),...
  [0,mid(tmax)],mid(a_k),opts);
app_time = toc - app_time;
% Plot approximate solution
if plt==1
  tspan=chebpts(length(nofft),[0,mid(tmax)]);
  mesh(x,tspan,(2*N+1)*real(ifft(ifftshift(nofft(tspan),2).')).')
%   mesh(k,tspan,abs(real(nofft(tspan)))) % plot Fourier coefficients
%   set(gca, 'ZScale', 'log')
  chebfunpref.setDefaults('factory');
  return
end

% Residual estimate
% finite dimensional term
a_kl = intval(chebcoeffs(nofft));
rescaleFactork = tmax/2;
a1_kl_verify = ChebDerCoeffs(a_kl)/rescaleFactork;
% d_kl=intval(zeros(size(a_kl)));
% for i=1:size(a_kl,1)-1
%    d_kl(i,:) = a1_kl_verify(i,:) + (convmat_core*(intval(1i)*k.*(a_kl(i,:).'))).';
% end
% i=size(a_kl,1);
% d_kl(end,:)=(convmat_core*(intval(1i)*k.*(a_kl(i,:).'))).';
d_kl = [a1_kl_verify;zeros(1,size(a1_kl_verify,2))]+(convmat_core*(intval(1i)*k.*(a_kl.'))).';

absdkl=abs(d_kl);
dklsum=sum(absdkl);
leftpart=norm(dklsum,2);


%Tail estimate
% c_n_part=intval(zeros(size(a_kl)));
% for i=1:size(a_kl,1)
%     c_n_part(i,:) = (convmat_beside*(intval(1i)*k.*(a_kl(i,:).'))).';
% end
c_n_part = (convmat_beside*(intval(1i)*k.*(a_kl.'))).';
abs_cn = abs(c_n_part);
cn_sum = sum(abs_cn);
rightpart = norm(cn_sum,2);

if c_inf>0
  part2 = intval(1i)*k.*(a_kl.');
  rightpart2 = c_inf*norm(sum(abs(part2)),2);
else
  rightpart2 = 0;
end
residual = leftpart+rightpart+rightpart2;

% Rigorous error estimate
error=exp(omega*tmax)*(initerr+((1-exp(-omega*tmax))/omega)*residual);
% error1=exp(omega*tmax)*initerr+((exp(omega*tmax)-1)/omega)*residual
rig_time = toc;

% data indexes
   y=intval(zeros(1,8));
   y(1)=N;
   y(2)=n;
   y(3)=tmax;
   y(4)=initerr;
   y(5)=residual;
   y(6)=error;
   y(7)=app_time;
   y(8)=rig_time;
chebfunpref.setDefaults('factory');
end